<?php

namespace App\Http\Controllers;

use App\Models\Proyek;
use Illuminate\Http\Request;
// [BARU] Impor Form Requests Anda
use App\Http\Requests\StoreProyekRequest;
use App\Http\Requests\UpdateProyekRequest;
use App\Models\Setting; // Pastikan ini ada untuk $globalSettings

class ProyekController extends Controller
{
    /**
     * Menampilkan daftar proyek dengan sorting dan pagination
     */
    public function index(Request $request)
    {
        $sortBy = $request->query('sort_by', 'created_at');
        $order = $request->query('order', 'desc');

        $validSorts = [
            'created_at', 'nama_proyek', 'nilai_kontrak', 
            'target_selesai', 'pic', 'status',
            'total_pengeluaran'
        ];

        if (!in_array($sortBy, $validSorts)) { $sortBy = 'created_at'; }
        if (!in_array($order, ['asc', 'desc'])) { $order = 'desc'; }

        $query = Proyek::withSum('pengeluarans as total_pengeluaran', 'total');

        if ($sortBy == 'sisa_anggaran') {
            $query->orderByRaw("COALESCE(nilai_kontrak, 0) - COALESCE(total_pengeluaran, 0) $order");
        } else {
            $query->orderBy($sortBy, $order);
        }

        $proyeks = $query->paginate(10)->withQueryString();
        
        // [TAMBAHAN] Ambil $globalSettings untuk layout
        $globalSettings = Setting::pluck('value', 'key')->all();

        return view('proyek.index', compact('proyeks', 'sortBy', 'order', 'globalSettings'));
    }

    /**
     * Menampilkan halaman detail proyek
     */
    public function show(Proyek $proyek)
    {
        $proyek->load('pengeluarans');
        
        // [TAMBAHAN] Ambil $globalSettings untuk layout
        $globalSettings = Setting::pluck('value', 'key')->all();

        return view('proyek.show', compact('proyek', 'globalSettings'));
    }

    /**
     * [DIUBAH] Memanggil private method baru
     */
    public function create()
    {
        $clients = $this->getDistinctClients();
        
        // [TAMBAHAN] Ambil $globalSettings untuk layout
        $globalSettings = Setting::pluck('value', 'key')->all();

        return view('proyek.create', compact('clients', 'globalSettings'));
    }

    /**
     * [DIUBAH] Controller jauh lebih bersih!
     * Validasi otomatis ditangani oleh StoreProyekRequest
     */
    public function store(StoreProyekRequest $request)
    {
        // Ambil data yang SUDAH divalidasi
        $data = $request->validated();
        
        // [LOGIKA LAMA] Tetap di sini, tapi lebih bersih
        if ($data['klien'] === 'TAMBAH_BARU_KLIEN_#') {
            $data['klien'] = $data['nama_klien_baru'];
        }
        
        $data['status'] = 'Sedang Berjalan';
        
        // Hapus 'nama_klien_baru' agar tidak masuk ke DB jika ada
        unset($data['nama_klien_baru']); 

        Proyek::create($data);

        return redirect()->route('proyek.index')
                         ->with('success', 'Proyek baru berhasil ditambahkan.');
    }

    /**
     * [DIUBAH] Memanggil private method baru
     */
    public function edit(Proyek $proyek)
    {
        $clients = $this->getDistinctClients();
        
        // [TAMBAHAN] Ambil $globalSettings untuk layout
        $globalSettings = Setting::pluck('value', 'key')->all();

        return view('proyek.edit', compact('proyek', 'clients', 'globalSettings'));
    }

    /**
     * [DIUBAH] Validasi otomatis ditangani oleh UpdateProyekRequest
     */
    public function update(UpdateProyekRequest $request, Proyek $proyek)
    {
        $data = $request->validated();

        if ($data['klien'] === 'TAMBAH_BARU_KLIEN_#') {
            $data['klien'] = $data['nama_klien_baru'];
        }
        
        unset($data['nama_klien_baru']);

        $proyek->update($data);

        return redirect()->route('proyek.index')
                         ->with('success', 'Data proyek berhasil diperbarui.');
    }

    
    public function destroy(Proyek $proyek)
    {
        // Dengan SoftDeletes di Model, ini akan MENGARSIPKAN, bukan menghapus
        $proyek->delete();

        return redirect()->route('proyek.index')
                         ->with('success', 'Proyek berhasil diarsipkan.');
    }
    
    public function tandaiSelesai(Proyek $proyek)
    {
        $proyek->update(['status' => 'Selesai']); // Sedikit lebih bersih

        return redirect()->route('proyek.index')->with('success', 'Status proyek berhasil diubah menjadi Selesai.');
    }

    /**
     * [METHOD BARU]
     * Helper untuk menghindari duplikasi query
     */
    private function getDistinctClients()
    {
        return Proyek::select('klien')
                     ->distinct()
                     ->orderBy('klien')
                     ->pluck('klien');
    }

    // FUNGSI dataKlien() SUDAH DIHAPUS DARI SINI
}