<?php

namespace App\Http\Controllers;

use App\Models\Proyek;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\StreamedResponse;
use Barryvdh\DomPDF\Facade\Pdf; 

class ReportController extends Controller
{
    /**
     * Menampilkan halaman laporan proyek dengan filter.
     */
    public function index(Request $request)
    {
        $all_proyeks = Proyek::select('id_proyek', 'nama_proyek')->orderBy('nama_proyek', 'asc')->get();

        $query = Proyek::with(['pengeluarans', 'tukangs'])
                       ->withSum('pengeluarans as total_pengeluaran', 'total')
                       // [PERBAIKAN] Ganti 'jumlah' -> 'total_pembayaran'
                       ->withSum('tukangs as total_upah', 'total_pembayaran');

        if ($request->filled('status') && $request->status != 'semua') {
            $query->where('status', $request->status);
        }
        if ($request->filled('tanggal_mulai') && $request->filled('tanggal_selesai')) {
            $query->whereBetween('tanggal_mulai', [$request->tanggal_mulai, $request->tanggal_selesai]);
        }
        if ($request->filled('proyek_id') && $request->proyek_id != 'semua') {
            $query->where('id_proyek', $request->proyek_id);
        }
        
        $proyeks = $query->orderBy('tanggal_mulai', 'desc')->get();

        $totalNilaiKontrak = $proyeks->sum('nilai_kontrak');
        $totalBiayaProyek = $proyeks->sum(function($proyek) {
            return ($proyek->total_pengeluaran ?? 0) + ($proyek->total_upah ?? 0);
        });

        $summary = [
            'total_nilai_kontrak' => $totalNilaiKontrak,
            'total_biaya_proyek'  => $totalBiayaProyek,
            'total_profit_loss'   => $totalNilaiKontrak - $totalBiayaProyek,
        ];

        return view('report.index', compact('proyeks', 'summary', 'all_proyeks'));
    }

    /**
     * Menangani ekspor ke CSV.
     */
    public function export($id_proyek)
    {
        $proyek = Proyek::with(['pengeluarans', 'tukangs'])
                       ->withSum('pengeluarans as total_pengeluaran', 'total')
                       // [PERBAIKAN] Ganti 'jumlah' -> 'total_pembayaran'
                       ->withSum('tukangs as total_upah', 'total_pembayaran')
                       ->findOrFail($id_proyek);
        
        $totalPengeluaran = $proyek->total_pengeluaran ?? 0;
        $totalUpah = $proyek->total_upah ?? 0;
        $nilaiKontrak = $proyek->nilai_kontrak ?? 0;
        $totalBiaya = $totalPengeluaran + $totalUpah;
        $profitLoss = $nilaiKontrak - $totalBiaya;

        $nama_file = 'Laporan Proyek - ' . preg_replace('/[^A-Za-z0-9\-]/', '', $proyek->nama_proyek ?? 'Tanpa Nama') . '.csv';

        $headers = [
            'Content-Type'        => 'text/csv; charset=utf-8',
            'Content-Disposition' => 'attachment; filename="' . $nama_file . '"',
        ];

        $callback = function() use ($proyek, $nilaiKontrak, $totalBiaya, $profitLoss, $totalPengeluaran, $totalUpah) {
            $file = fopen('php://output', 'w');
            fputs($file, (chr(0xEF) . chr(0xBB) . chr(0xBF))); // BOM for UTF-8

            // Header Info
            fputcsv($file, ['LAPORAN PROYEK:', $proyek->nama_proyek ?? 'Tanpa Nama']);
            fputcsv($file, ['KLIEN:', $proyek->klien ?? '-']);
            fputcsv($file, []); 

            // Summary
            fputcsv($file, ['RINGKASAN']);
            fputcsv($file, ['NILAI KONTRAK', 'TOTAL BIAYA', 'PROFIT/LOSS']);
            fputcsv($file, [$nilaiKontrak, $totalBiaya, $profitLoss]);
            fputcsv($file, []); 

            // Rincian Material/Pengeluaran
            fputcsv($file, ['RINCIAN PEMBAYARAN MATERIAL']);
            fputcsv($file, ['Tanggal', 'Nama Toko', 'Deskripsi', 'Kategori', 'Jumlah', 'Status Pembayaran']);
            if ($proyek->pengeluarans && $proyek->pengeluarans->isNotEmpty()) {
                foreach ($proyek->pengeluarans as $p) {
                    fputcsv($file, [
                        isset($p->tanggal_struk) ? \Carbon\Carbon::parse($p->tanggal_struk)->format('d/m/Y') : '-', 
                        $p->toko ?? '-', 
                        $p->items ?? '-', 
                        $p->kategori ?? '-', 
                        $p->total ?? 0,
                        $p->status_bayar ?? '-' 
                    ]);
                }
                fputcsv($file, ['TOTAL PENGELUARAN MATERIAL', '', '', '', $totalPengeluaran, '']);
            } else {
                fputcsv($file, ['- Tidak ada data -', '', '', '', '', '']);
            }
            fputcsv($file, []); 

            // Rincian Tukang
            fputcsv($file, ['RINCIAN PEMBAYARAN TUKANG']);
            // [PERBAIKAN] Sesuaikan header CSV
            fputcsv($file, ['Nama Mandor', 'Tgl. Jatuh Tempo', 'Jumlah (Rp)', 'Status Pembayaran']); 
            if ($proyek->tukangs && $proyek->tukangs->isNotEmpty()) {
                foreach ($proyek->tukangs as $t) {
                    fputcsv($file, [
                        // [PERBAIKAN] Sesuaikan data CSV
                        $t->nama_mandor ?? $t->nama_tukang ?? '-', // Tampilkan mandor, fallback ke tukang
                        isset($t->tanggal_jatuh_tempo) ? \Carbon\Carbon::parse($t->tanggal_jatuh_tempo)->format('d/m/Y') : '-', 
                        $t->total_pembayaran ?? 0,
                        $t->status_pembayaran ?? '-'
                    ]);
                }
                fputcsv($file, ['TOTAL UPAH TUKANG', '', $totalUpah, '']);
            } else {
                fputcsv($file, ['- Tidak ada data -', '', '', '']);
            }

            fclose($file);
        };

        return new StreamedResponse($callback, 200, $headers);
    }
    
    
    /**
     * Menangani ekspor ke PDF.
     */
    public function exportPdf($id_proyek)
    {
        $proyek = Proyek::with(['pengeluarans', 'tukangs'])
                       ->withSum('pengeluarans as total_pengeluaran', 'total')
                       // [PERBAIKAN] Ganti 'jumlah' -> 'total_pembayaran'
                       ->withSum('tukangs as total_upah', 'total_pembayaran')
                       ->findOrFail($id_proyek);
        
        $totalPengeluaran = $proyek->total_pengeluaran ?? 0;
        $totalUpah = $proyek->total_upah ?? 0;
        $nilaiKontrak = $proyek->nilai_kontrak ?? 0;
        $totalBiaya = $totalPengeluaran + $totalUpah;
        $profitLoss = $nilaiKontrak - $totalBiaya;

        $data = [
            'proyek' => $proyek,
            'totalBiaya' => $totalBiaya,
            'profitLoss' => $profitLoss,
            'tanggalCetak' => now()->translatedFormat('d F Y')
        ];

        try {
            $pdf = Pdf::loadView('report.pdf_template', $data);
        
            $nama_file = 'Laporan PDF - ' . preg_replace('/[^A-Za-z0-9\-]/', '', $proyek->nama_proyek ?? 'Tanpa Nama') . '.pdf';
            
            return $pdf->download($nama_file);

        } catch (\Exception $e) {
            \Log::error('Error generating PDF: '. $e->getMessage());
            return redirect()->back()->with('error', 'Gagal membuat file PDF: '. $e->getMessage());
        }
    }
}
