<?php

namespace App\Http\Controllers;

use App\Models\Task;
use App\Models\Proyek;
use App\Models\User;     // Tetap diperlukan untuk Auth::id() dan relasi 'creator'
use App\Models\Anggota;  // [DIUBAH] Tambahkan model Anggota
// use App\Models\Team;  // [DIHAPUS] Model Team tidak lagi digunakan

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class TaskController extends Controller
{
    /**
     * Menampilkan halaman monitor (index)
     * Disederhanakan: Hanya Admin yang bisa melihat halaman ini.
     * Logika filter tim dihapus.
     */
    public function index(Request $request)
    {
        // [DIUBAH] Relasi 'assignee' diganti menjadi 'anggota'
        $query = Task::with('proyek', 'anggota'); 

        // Asumsi hanya admin yang bisa mengakses index task
        if (Auth::user()->role != 'admin') {
             abort(403, 'Akses ditolak. Hanya Admin.');
        }

        // [DIHAPUS] Logika filter berdasarkan Team
        // [DIHAPUS] Logika 'else' untuk non-admin

        // Paginate hasil
        $tasks = $query->latest()->paginate(10);
        
        // [DIHAPUS] $teams tidak lagi diambil
        
        // [DIUBAH] 'teams' dihapus dari compact()
        return view('tasks.index', compact('tasks'));
    }

    /**
     * Menampilkan form untuk membuat task baru (Hanya Admin)
     */
    public function create()
    {
        $proyeks = Proyek::where('status', '!=', 'Selesai')->orderBy('nama_proyek')->get();
        // [DIUBAH] Ambil data 'anggota', bukan 'users'
        $anggota = Anggota::orderBy('nama')->get(); 
        
        // [DIUBAH] Kirim 'anggota' ke view
        return view('tasks.create', compact('proyeks', 'anggota'));
    }

    /**
     * Menyimpan task baru (Hanya Admin)
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'proyek_id' => 'required|exists:proyeks,id_proyek',
            'anggota_id' => 'required|exists:anggota,id', // [DIUBAH] Validasi ke tabel anggota
            'priority' => 'required|in:Low,Medium,High',
            'due_date' => 'nullable|date',
        ]);

        Task::create([
            'title' => $request->title,
            'description' => $request->description,
            'proyek_id' => $request->proyek_id,
            'anggota_id' => $request->anggota_id, // [DIUBAH] Menyimpan anggota_id
            'created_by_user_id' => Auth::id(), 
            'priority' => $request->priority,
            'due_date' => $request->due_date,
            'status' => 'To Do',
        ]);

        return redirect()->route('tasks.index')->with('success', 'Task baru berhasil dibuat dan ditugaskan.');
    }

    /**
     * Menampilkan halaman detail task
     */
    public function show(Task $task)
    {
        // [DIUBAH] Disederhanakan hanya untuk admin
        if (Auth::user()->role != 'admin') {
            abort(403, 'Akses ditolak');
        }
        
        return redirect()->route('tasks.edit', $task); // Sementara redirect ke edit
    }

    /**
     * Menampilkan form edit task
     */
    public function edit(Task $task)
    {
        // [DIUBAH] Disederhanakan hanya untuk admin
        if (Auth::user()->role != 'admin') {
             abort(403, 'Akses ditolak');
        }

        $proyeks = Proyek::orderBy('nama_proyek')->get();
        // [DIUBAH] Ambil data 'anggota', bukan 'users'
        $anggota = Anggota::orderBy('nama')->get();
        
        // [DIUBAH] Kirim 'anggota' ke view
        return view('tasks.edit', compact('task', 'proyeks', 'anggota'));
    }

    /**
     * Update task
     * [DIUBAH] Disederhanakan, hanya Admin yang bisa update.
     */
    public function update(Request $request, Task $task)
    {
        $user = Auth::user();
        
        // Hanya Admin yang bisa update
        if ($user->role == 'admin') {
            $request->validate([
                'title' => 'required|string|max:255',
                'description' => 'nullable|string',
                'proyek_id' => 'required|exists:proyeks,id_proyek',
                'anggota_id' => 'required|exists:anggota,id', // [DIUBAH] Validasi ke tabel anggota
                'status' => 'required|in:To Do,In Progress,Done',
                'priority' => 'required|in:Low,Medium,High',
                'due_date' => 'nullable|date',
            ]);

            $data = $request->except(['_token', '_method']);
            
            if ($request->status == 'Done' && $task->status != 'Done') {
                $data['completed_at'] = now();
            }

            $task->update($data);
            
        } else {
            // [DIHAPUS] Logika 'else if' untuk non-admin
            return redirect()->route('tasks.index')->with('error', 'Anda tidak diizinkan mengupdate task ini.');
        }

        return redirect()->route('tasks.index')->with('success', 'Task berhasil diperbarui.');
    }
    public function updateStatus(Request $request, Task $task)
    {
        // Asumsi hanya admin yang bisa update
        if(Auth::user()->role != 'admin') { 
            return response()->json(['success' => false, 'message' => 'Akses ditolak.'], 403);
        }

        $request->validate([
            'status' => 'required|in:To Do,In Progress,Done',
        ]);

        $data = ['status' => $request->status];

        // Jika status baru adalah 'Done' dan status lama BUKAN 'Done', 
        // catat waktu selesainya.
        if ($request->status == 'Done' && $task->status != 'Done') {
            $data['completed_at'] = now();
        } 
        // Jika task dikembalikan dari 'Done' ke status lain
        else if ($request->status != 'Done' && $task->status == 'Done') {
            $data['completed_at'] = null;
        }

        $task->update($data);

        // Kirim kembali data task yang sudah di-update (termasuk relasi)
        return response()->json([
            'success' => true,
            'message' => 'Status task berhasil diperbarui.',
            'task' => $task->load('anggota', 'proyek') 
        ]);
    }
    public function destroy(Task $task)
    {
        // Tambahkan pengecekan role admin
        if(Auth::user()->role != 'admin') { 
            abort(403, 'Akses ditolak'); 
        }

        $task->delete(); 
        return redirect()->route('tasks.index')->with('success', 'Task berhasil dihapus.');
    }
}