<?php

namespace App\Http\Controllers;

use App\Models\Team;
use App\Models\User;
use App\Models\Task; // <-- [TAMBAHKAN] Import Task
use Illuminate\Http\Request;

class TeamController extends Controller
{
    /**
     * [DIUBAH] Menampilkan Papan Kanban (Progress Board)
     * Menggantikan tabel 'teams'
     */
    public function index()
    {
        // Ambil semua task, urutkan agar yang terbaru di atas
        $tasks = Task::with('anggota', 'proyek')
                     ->orderBy('updated_at', 'desc')
                     ->get();

        // Kelompokkan task berdasarkan statusnya
        $tasksByStatus = [
            'To Do' => $tasks->where('status', 'To Do'),
            'In Progress' => $tasks->where('status', 'In Progress'),
            'Done' => $tasks->where('status', 'Done'),
        ];

        return view('teams.index', compact('tasksByStatus'));
    }

    public function create()
    {
        // Halaman ini tidak lagi dipakai oleh index,
        // tapi kita biarkan logikanya
        return view('teams.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:teams,name',
            'description' => 'nullable|string',
        ]);

        Team::create($request->all());

        return redirect()->route('teams.index')->with('success', 'Tim baru berhasil dibuat.');
    }

    public function show(Team $team)
    {
        $team->load('users');
        return view('teams.show', compact('team'));
    }

    public function edit(Team $team)
    {
        $users = User::orderBy('name')->get();
        $teamUserIds = $team->users->pluck('id')->toArray();
        
        return view('teams.edit', compact('team', 'users', 'teamUserIds'));
    }

    public function update(Request $request, Team $team)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:teams,name,' . $team->id,
            'description' => 'nullable|string',
            'users' => 'nullable|array',
            'users.*' => 'exists:users,id'
        ]);

        $team->update($request->only('name', 'description'));

        if ($request->has('users')) {
            $team->users()->sync($request->input('users', []));
        } else {
            $team->users()->sync([]);
        }

        return redirect()->route('teams.index')->with('success', 'Data tim berhasil diperbarui.');
    }

    public function destroy(Team $team)
    {
        $team->users()->detach();
        $team->delete();
        
        return redirect()->route('teams.index')->with('success', 'Tim berhasil dihapus.');
    }
}