<?php

namespace App\Http\Controllers;

use App\Models\Tukang;
use App\Models\Proyek;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule; // [TAMBAHAN] Diperlukan untuk validasi

class TukangController extends Controller
{
    /**
     * Menampilkan daftar semua data pembayaran tukang.
     * [MODIFIKASI] Menambahkan logika filter dan data untuk filter.
     */
    public function index(Request $request) // [MODIFIKASI] Tambahkan Request
    {
        // Ambil query dasar
        $query = Tukang::with('proyek')->latest();

        // [TAMBAHAN] Logika filter
        if ($request->filled('id_proyek')) {
            $query->where('id_proyek', $request->id_proyek);
        }
        if ($request->filled('nama_mandor')) {
            $query->where('nama_mandor', $request->nama_mandor);
        }

        // [MODIFIKASI] Paginate hasil query dan tambahkan query string filter
        $pembayaranTukangs = $query->paginate(10)->appends($request->query());

        // [TAMBAHAN] Data untuk dropdown filter
        $proyeks = Proyek::orderBy('nama_proyek')->get();
        $mandors = Tukang::select('nama_mandor')->distinct()->orderBy('nama_mandor')->get();

        return view('tukang.index', [
            'tukangs' => $pembayaranTukangs,
            'proyeks' => $proyeks,
            'mandors' => $mandors,
        ]);
    }

    /**
     * Menampilkan form untuk menambah data pembayaran baru.
     * [MODIFIKASI] Menambahkan data mandor yang sudah ada.
     */
    public function create()
    {
        $proyeks = Proyek::where('status', '!=', 'Selesai')->orderBy('nama_proyek')->get();
        // [TAMBAHAN] Ambil daftar mandor unik
        $mandors = Tukang::select('nama_mandor')->distinct()->orderBy('nama_mandor')->get();
        
        return view('tukang.create', compact('proyeks', 'mandors'));
    }

    /**
     * Menyimpan data pembayaran baru ke database.
     * [MODIFIKASI] Menyesuaikan validasi dan logika penyimpanan nama_mandor.
     */
    public function store(Request $request)
    {
        // [MODIFIKASI] Validasi untuk mandor baru
        $request->validate([
            'id_proyek' => 'required|exists:proyeks,id_proyek',
            'nama_mandor' => 'required|string|max:255',
            'nama_mandor_baru' => [
                'nullable',
                'string',
                'max:255',
                // Wajib diisi HANYA JIKA nama_mandor adalah 'TAMBAH_BARU'
                Rule::requiredIf($request->nama_mandor == 'TAMBAH_BARU_MANDOR_#'), 
                // [Opsional] Pastikan unik jika diisi
                // 'unique:tukangs,nama_mandor' 
            ],
            'total_pembayaran' => 'required|numeric|min:0',
            'tanggal_jatuh_tempo' => 'required|date',
            'bukti_absensi' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:2048',
            'tanggal_pembayaran' => 'nullable|date',
            'bukti_pembayaran' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:2048',
        ]);

        $data = $request->except(['bukti_absensi', 'bukti_pembayaran', 'nama_mandor_baru']);
        
        // [TAMBAHAN] Logika untuk menyimpan mandor baru atau yang sudah ada
        if ($request->nama_mandor == 'TAMBAH_BARU_MANDOR_#' && $request->filled('nama_mandor_baru')) {
            $data['nama_mandor'] = $request->nama_mandor_baru;
        } else {
            $data['nama_mandor'] = $request->nama_mandor;
        }

        $data['status_pembayaran'] = 'Belum Lunas'; // Status default

        // Handle File Upload Bukti Absensi
        if ($request->hasFile('bukti_absensi')) {
            $path = $request->file('bukti_absensi')->store('bukti_absensi', 'public');
            $data['bukti_absensi'] = $path;
        }

        // Handle File Upload Bukti Pembayaran
        if ($request->hasFile('bukti_pembayaran')) {
            $path = $request->file('bukti_pembayaran')->store('bukti_pembayaran', 'public');
            $data['bukti_pembayaran'] = $path;
        }
        
        Tukang::create($data);

        return redirect()->route('tukang.index')
                         ->with('success', 'Data pembayaran tukang berhasil ditambahkan.');
    }

    /**
     * Menampilkan form untuk mengedit data pembayaran.
     * [MODIFIKASI] Menambahkan data mandor yang sudah ada.
     */
    public function edit(Tukang $tukang)
    {
        $proyeks = Proyek::orderBy('nama_proyek')->get();
        // [TAMBAHAN] Ambil daftar mandor unik
        $mandors = Tukang::select('nama_mandor')->distinct()->orderBy('nama_mandor')->get();
        
        return view('tukang.edit', compact('tukang', 'proyeks', 'mandors'));
    }

    /**
     * Memperbarui data pembayaran di database.
     * [MODIFIKASI] Menyesuaikan validasi dan logika update nama_mandor.
     */
    public function update(Request $request, Tukang $tukang)
    {
        // [MODIFIKASI] Validasi untuk mandor baru
        $request->validate([
            'id_proyek' => 'required|exists:proyeks,id_proyek',
            'nama_mandor' => 'required|string|max:255',
            'nama_mandor_baru' => [
                'nullable',
                'string',
                'max:255',
                Rule::requiredIf($request->nama_mandor == 'TAMBAH_BARU_MANDOR_#'),
            ],
            'total_pembayaran' => 'required|numeric|min:0',
            'tanggal_jatuh_tempo' => 'required|date',
            'status_pembayaran' => 'required|in:Lunas,Belum Lunas',
            'bukti_absensi' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:2048',
            'tanggal_pembayaran' => 'nullable|date',
            'bukti_pembayaran' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:2048',
        ]);

        $data = $request->except(['bukti_absensi', 'bukti_pembayaran', 'nama_mandor_baru']);
        
        // [TAMBAHAN] Logika untuk update mandor baru atau yang sudah ada
        if ($request->nama_mandor == 'TAMBAH_BARU_MANDOR_#' && $request->filled('nama_mandor_baru')) {
            $data['nama_mandor'] = $request->nama_mandor_baru;
        } else {
            $data['nama_mandor'] = $request->nama_mandor;
        }

        // Handle Update Bukti Absensi
        if ($request->hasFile('bukti_absensi')) {
            if ($tukang->bukti_absensi) {
                Storage::disk('public')->delete($tukang->bukti_absensi);
            }
            $path = $request->file('bukti_absensi')->store('bukti_absensi', 'public');
            $data['bukti_absensi'] = $path;
        }

        // Handle Update Bukti Pembayaran
        if ($request->hasFile('bukti_pembayaran')) {
            if ($tukang->bukti_pembayaran) {
                Storage::disk('public')->delete($tukang->bukti_pembayaran);
            }
            $path = $request->file('bukti_pembayaran')->store('bukti_pembayaran', 'public');
            $data['bukti_pembayaran'] = $path;
        }

        if ($request->status_pembayaran == 'Lunas' && empty($request->tanggal_pembayaran)) {
            $data['tanggal_pembayaran'] = now();
        }

        $tukang->update($data);

        return redirect()->route('tukang.index')
                         ->with('success', 'Data pembayaran tukang berhasil diperbarui.');
    }

    /**
     * Menghapus data pembayaran dari database.
     */
    public function destroy(Tukang $tukang)
    {
        if ($tukang->bukti_absensi) {
            Storage::disk('public')->delete($tukang->bukti_absensi);
        }
        if ($tukang->bukti_pembayaran) {
            Storage::disk('public')->delete($tukang->bukti_pembayaran);
        }

        $tukang->delete();
        return redirect()->route('tukang.index')
                         ->with('success', 'Data pembayaran tukang berhasil dihapus.');
    }
    
    /**
     * Menandai upah tukang sebagai lunas.
     * [MODIFIKASI] Menerima Request dan file upload dari modal.
     */
    public function tandaiLunas(Request $request, Tukang $tukang)
    {
        // [TAMBAHAN] Validasi untuk file yang diupload dari modal
        $request->validate([
            'bukti_absensi' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:2048',
            'bukti_pembayaran' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:2048',
        ]);

        $data = []; // Siapkan data untuk update

        // [TAMBAHAN] Handle Update Bukti Absensi (jika diupload via modal)
        if ($request->hasFile('bukti_absensi')) {
            if ($tukang->bukti_absensi) {
                Storage::disk('public')->delete($tukang->bukti_absensi);
            }
            $path = $request->file('bukti_absensi')->store('bukti_absensi', 'public');
            $data['bukti_absensi'] = $path;
        }

        // [TAMBAHAN] Handle Update Bukti Pembayaran (jika diupload via modal)
        if ($request->hasFile('bukti_pembayaran')) {
            if ($tukang->bukti_pembayaran) {
                Storage::disk('public')->delete($tukang->bukti_pembayaran);
            }
            $path = $request->file('bukti_pembayaran')->store('bukti_pembayaran', 'public');
            $data['bukti_pembayaran'] = $path;
        }

        // Set status dan tanggal
        $data['status_pembayaran'] = 'Lunas';
        $data['tanggal_pembayaran'] = now();
        
        $tukang->update($data); // Update data tukang

        return redirect()->route('tukang.index')->with('success', 'Pembayaran berhasil ditandai lunas.');
    }
}